#include "Utilities/Configuration/interface/Architecture.h"
#include "ClassReuse/GeomVector/interface/GlobalPoint.h"

#include "CommonDet/BasicDet/interface/RecHit.h"
#include "CommonDet/DetLayout/interface/DetLayer.h"
#include "CommonDet/DetLayout/interface/BarrelDetLayer.h"
#include "CommonDet/DetLayout/interface/ForwardDetLayer.h"

#include "TrackerReco/TkHitPairs/interface/OrderedHitPair.h"
#include "TrackerReco/TkHitTriplets/interface/ThirdHitRZPredictionWithHelix.h"

#include "TrackerReco/TkMSParametrization/interface/MultipleScatteringParametrisation.h"

#include "TrackerReco/PixelTrackFit/interface/CircleFromThreePoints.h"

#include <Utilities/Notification/interface/TimingReport.h>

#undef Debug

#include <fstream>

inline float sqr(float x) { return x*x; }

/*****************************************************************************/
#include "MagneticField/BaseMagneticField/interface/MagneticField.h"
#include "ClassReuse/GeomVector/interface/GlobalPoint.h"

ThirdHitRZPredictionWithHelix::ThirdHitRZPredictionWithHelix
  (float originRBound, float ptMin, GlobalPoint inner, GlobalPoint outer)
{
#ifdef Debug
 printOut("constructor");
#endif

 Bz = fabs(MagneticField::inInverseGeV(GlobalPoint(0,0,0)).z());

 r0 = originRBound;
 rm = ptMin / Bz;
 g1 = inner;
 g2 = outer;

 p1 = Global2DVector(g1.x(), g1.y());
 p2 = Global2DVector(g2.x(), g2.y());

 dif = p1 - p2;

 // Prepare circles of minimal pt (rm) and cylinder of origin (r0)
 keep = true;
 arc_0m = findArcIntersection(findMinimalCircles (rm),
                              findTouchingCircles(r0), keep);

#ifdef Debug
 fprintf(stderr,
   "   [PredictionWithHelix]  r0=%.2f rm=%.2f\n",r0,rm);
 fprintf(stderr,
   "   [PredictionWithHelix]  p1=(%.2f %.2f %.2f) p2=(%.2f %.2f %.2f)\n",
   g1.x(),g1.y(),g1.z(),
   g2.x(),g2.y(),g2.z());

 pair<float,float> arc_m = findMinimalCircles (rm);
 fprintf(stderr,
   "   [PredictionWithHelix]  arc_m =(%.3f %.3f)\n",
   arc_m.first,arc_m.second);

 pair<float,float> arc_0 = findTouchingCircles(r0);
 fprintf(stderr,
   "   [PredictionWithHelix]  arc_0 =(%.3f %.3f)\n",
   arc_0.first,arc_0.second);

 fprintf(stderr,
   "   [PredictionWithHelix]  arc_0m=(%.3f %.3f)\n",
   arc_0m.first,arc_0m.second);
#endif
}

/*****************************************************************************/
ThirdHitRZPredictionWithHelix::~ThirdHitRZPredictionWithHelix()
{
#ifdef Debug
 printOut("destructor");
#endif
}

#ifdef Debug
/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::printOut(char *text)
{
 cerr << "   [PredictionWithHelix] " << text << endl;
}
#endif

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::invertCircle(Global2DVector& c,float& r)
{
  float s = dif.mag2() / ((c - p1).mag2() - sqr(r));

  c = p1 + (c - p1)*s;
  r *= fabsf(s);
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::invertPoint(Global2DVector& p)
{
  float s = dif.mag2() / (p - p1).mag2();

  p = p1 + (p - p1)*s;
}

/*****************************************************************************/
pair<float,float> ThirdHitRZPredictionWithHelix::findMinimalCircles(float r)
{ 
  pair<float,float> a(0.,0.);

  if(dif.mag2() <  2 * sqr(r))
    a = pair<float,float>( dif.phi(),
                           0.5*acos(1 - 0.5 * dif.mag2()/sqr(r)) );

  return a;
}

/*****************************************************************************/
pair<float,float> ThirdHitRZPredictionWithHelix::findTouchingCircles(float r)
{
  Global2DVector c(0.,0.);
  invertCircle(c,r);

  pair<float,float> a(0.,0.);
  a = pair<float,float>( (c - p2).phi(),
                          0.5*acos(1 - 2*sqr(r)/(c - p2).mag2()) );

  return a;
}

/*****************************************************************************/
pair<float,float> ThirdHitRZPredictionWithHelix::findArcIntersection
  (pair<float,float> a, pair<float,float> b, bool& keep)
{ 
  // spin closer
  while(b.first < a.first - M_PI) b.first += 2*M_PI;
  while(b.first > a.first + M_PI) b.first -= 2*M_PI;

  float min,max;

  if(a.first - a.second > b.first - b.second)
    min = a.first - a.second;
  else
  { min = b.first - b.second; keep = false; }

  if(a.first + a.second < b.first + b.second)
    max = a.first + a.second;
  else
  { max = b.first + b.second; keep = false; }
  
  pair<float,float> c(0.,0.);

  if(min < max)
  {
    c.first  = 0.5*(max + min);
    c.second = 0.5*(max - min);
  }

  return c;
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::fitParabola
  (const float x[3], const float y[3], float par[3])
{ 
  float s2 = sqr(x[0]) * (y[1] - y[2]) +
             sqr(x[1]) * (y[2] - y[0]) +
             sqr(x[2]) * (y[0] - y[1]);
  
  float s1 =     x[0]  * (y[1] - y[2]) +
                 x[1]  * (y[2] - y[0]) +
                 x[2]  * (y[0] - y[1]);

  float s3 = (x[0] - x[1]) * (x[1] - x[2]) * (x[2] - x[0]);
  float s4 = x[0]*x[1]*y[2] * (x[0] - x[1]) +
             x[0]*y[1]*x[2] * (x[2] - x[0]) +
             y[0]*x[1]*x[2] * (x[1] - x[2]);

  par[2] =  s1 / s3; // a2
  par[1] = -s2 / s3; // a1
  par[0] = -s4 / s3; // a0
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::findRectangle
  (const float x[3], const float y[3], const float par[3],
   float phi[2],float z[2])
{ 
  // Initial guess
  phi[0] = x[0] <? x[1]; z[0] = y[0] <? y[2];
  phi[1] = x[0] >? x[2]; z[1] = y[0] >? y[2];

  // Extremum: position and value
  float xe = -par[1]/(2*par[2]);
  float ye = par[0] - sqr(par[1])/(4*par[2]);

  // Check if extremum is inside the phi range
  if(phi[0] < xe  && xe < phi[1])
  {
    if(ye < z[0]) z[0] = ye;
    if(ye > z[1]) z[1] = ye;
  }
}

/*****************************************************************************/
float ThirdHitRZPredictionWithHelix::areaParallelogram
  (const Global2DVector& a, const Global2DVector& b)
{
  return a.x() * b.y() - a.y() * b.x();
}

/*****************************************************************************/
float ThirdHitRZPredictionWithHelix::angleRatio
  (const Global2DVector& p3, const Global2DVector& c)
{
  float rad2 = (p1 - c).mag2();

  float a12 = asin(fabsf(areaParallelogram(p1 - c, p2 - c)) / rad2);
  float a23 = asin(fabsf(areaParallelogram(p2 - c, p3 - c)) / rad2);

  return a23/a12;
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::spinCloser(float phi[3])
{
  while(phi[1] < phi[0] - M_PI) phi[1] += 2*M_PI;
  while(phi[1] > phi[0] + M_PI) phi[1] -= 2*M_PI;

  while(phi[2] < phi[1] - M_PI) phi[2] += 2*M_PI;
  while(phi[2] > phi[1] + M_PI) phi[2] -= 2*M_PI;
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::calculateRangesBarrel
  (float r3, float phi[2],float z[2], bool keep)
{
  pair<float,float> arc_all =
    findArcIntersection(arc_0m, findTouchingCircles(r3), keep);

  if(arc_all.second != 0.)
  {
//    TimeMe tm1("ThirdHitRZPredictionWithHelix::calculateRangesBarrel", true);

    Global2DVector c3(0.,0.); // barrel at r3
    invertCircle(c3,r3);      // inverted

    float angle[3];           // prepare angles
    angle[0] = arc_all.first - arc_all.second;
    angle[1] = arc_all.first;
    angle[2] = arc_all.first + arc_all.second;

    float phi3[3], z3[3];
    Global2DVector delta = c3 - p2;

    for(int i=0; i<3; i++)
    {
      Global2DVector vec(cos(angle[i]), sin(angle[i])); // unit vector
      float lambda = delta*vec - sqrt(sqr(delta*vec) - delta*delta + sqr(r3));

      Global2DVector p3 = p2 + lambda * vec;  // inverted third hit 
      invertPoint(p3);                        // third hit
      phi3[i] = p3.phi();                     // phi of third hit

      float ratio;

      if(keep && i==1)
      { // Straight line
        ratio = (p2 - p3).mag() / (p1 - p2).mag();
      }
      else
      { // Circle
        Global2DVector c = p2 - vec * (vec * (p2 - p1)); // inverted antipodal
        invertPoint(c);                                  // antipodal
        c = 0.5*(p1 + c);                                // center

        ratio = angleRatio(p3,c);
      }

      z3[i] = g2.z() + (g2.z() - g1.z()) * ratio;        // z of third hit
    }

    spinCloser(phi);

    // Parabola on phi - z
    float par[3];
    fitParabola  (phi3,z3, par);
    findRectangle(phi3,z3, par, phi,z);

#ifdef Debug
    ofstream outFile("out/p1p2.dat");
    outFile << g1.x() << " " << g1.y() << " " << g1.z() << endl;
    outFile << g2.x() << " " << g2.y() << " " << g2.z() << endl;
    outFile.close();
#endif
  }
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::calculateRangesForward
  (float z3, float phi[2],float r[2], bool keep)
{
  float angle[3];           // prepare angles
  angle[0] = arc_0m.first - arc_0m.second;
  angle[1] = arc_0m.first;
  angle[2] = arc_0m.first + arc_0m.second;

  float ratio = (z3 - g2.z()) / (g2.z() - g1.z());

  if(0 < ratio  && ratio < 3)
  {
//    TimeMe tm2("ThirdHitRZPredictionWithHelix::calculateRangesForward", true);
    float phi3[3], r3[3];

    for(int i=0; i<3; i++)
    {
      Global2DVector p3;

      if(keep && i==1)
      { // Straight line
        p3 = p2 + ratio * (p2 - p1);
      }
      else
      { // Circle
        Global2DVector vec(cos(angle[i]), sin(angle[i])); // unit vector
  
        Global2DVector c = p2 - vec * (vec * (p2 - p1));  // inverted antipodal
        invertPoint(c);                                   // antipodal
        c = 0.5*(p1 + c);                                 // center

        float rad2 = (p1 - c).mag2();

        float a12 = asin(areaParallelogram(p1 - c, p2 - c) / rad2);
        float a23 = ratio * a12;

        p3 = c + Global2DVector((p2-c).x()*cos(a23) - (p2-c).y()*sin(a23),
                                (p2-c).x()*sin(a23) + (p2-c).y()*cos(a23));
      }

      phi3[i] = p3.phi();
      r3[i]   = p3.mag();
    }

    spinCloser(phi3);

    // Parabola on phi - z
    float par[3];
    fitParabola  (phi3,r3, par);
    findRectangle(phi3,r3, par, phi,r);

    ofstream outFile("out/p1p2.dat");
    outFile << g1.x() << " " << g1.y() << " " << g1.z() << endl;
    outFile << g2.x() << " " << g2.y() << " " << g2.z() << endl;
    outFile.close();
  }
#ifdef Debug
  else
   fprintf(stderr,"   [PredictionWithHelix]  ratio < 0 || 3 > ratio\n");
#endif
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::calculateRanges
  (float rz3, float phi[2],float rz[2])
{
  // Clear
  phi[0] = 0.; rz[0] = 0.;
  phi[1] = 0.; rz[1] = 0.;

  // Calculate
  if(theBarrel) calculateRangesBarrel (rz3, phi,rz, keep);
           else calculateRangesForward(rz3, phi,rz, keep);

#ifdef Debug
  fprintf(stderr,
    "   [PredictionWithHelix]  rz3=%5.2f phi(%.2f %.2f), rz(%.2f %.2f)\n",
    rz3, phi[0],phi[1], rz[0],rz[1]);
#endif
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::getRanges
  (const DetLayer *layer, float phi[],float rz[])
{
  TimeMe tm3("ThirdHitRZPredictionWithHelix::getRanges:Layer", true);

  theLayer = layer;
  
  if (layer) initLayer(layer);

#ifdef Debug
  fprintf(stderr, "   [PredictionWithHelix]  getRanges:Layer %s\n",
          theBarrel ? "barrel" : "forward");
#endif

  float phi_inner[2],rz_inner[2];
  calculateRanges(theDetRange.min(), phi_inner,rz_inner);

  float phi_outer[2],rz_outer[2];
  calculateRanges(theDetRange.max(), phi_outer,rz_outer);

  phi[0] = min(phi_inner[0],phi_outer[0]);
  phi[1] = max(phi_inner[1],phi_outer[1]);

   rz[0] = min( rz_inner[0], rz_outer[0]);
   rz[1] = max( rz_inner[1], rz_outer[1]);

#ifdef Debug
  fprintf(stderr,
          "   [PredictionWithHelix]! phi(%.3f %.3f), rz(%.3f %.3f)\n",
    phi[0],phi[1],
    rz[0],rz[1]);
#endif
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::getRanges
  (float rz3, float phi[],float rz[])
{
#ifdef Debug
  printOut("getRanges:Hit");
#endif
  TimeMe tm4("ThirdHitRZPredictionWithHelix::getRanges:Hit", true);

  calculateRanges(rz3, phi,rz);
}

/*****************************************************************************/
float ThirdHitRZPredictionWithHelix::getTheta0(float p)
{
 // 300 um silicon, pion
 float m_pi = 0.13957018;
 float beta = p / sqrt(sqr(p) + sqr(m_pi));

 return 0.602e-3 / (beta * p);
}

/*****************************************************************************/
float ThirdHitRZPredictionWithHelix::multipleScattering
  (const float &pt, const GlobalPoint& g3)
{
  TimeMe tm5("ThirdHitRZPredictionWithHelix::multipleScattering1", true);

  PixelRecoPointRZ rz1(g1.perp(), g1.z());
  PixelRecoPointRZ rz3(g3.perp(), g3.z());

  MultipleScatteringParametrisation msp(theLayer); 
  return msp(pt, rz1,rz3);
}

/*****************************************************************************/
float ThirdHitRZPredictionWithHelix::multipleScattering
  (const float &pt, const float& cotTheta)
{
  TimeMe tm6("ThirdHitRZPredictionWithHelix::multipleScattering2", true);

  MultipleScatteringParametrisation msp(theLayer);
  return msp(pt, cotTheta);
}

/*****************************************************************************/
bool ThirdHitRZPredictionWithHelix::isCompatible(GlobalPoint g3)
{
  TimeMe tm7("ThirdHitRZPredictionWithHelix::isCompatible", true);

  CircleFromThreePoints circle(g1,g2,g3);
  Global2DVector c (circle.center().x(), circle.center().y());
  Global2DVector p3(g3.x(), g3.y());

  float rad2 = (p1 - c).mag2();
  float a12 = asin(fabsf(areaParallelogram(p1 - c, p2 - c)) / rad2);
  float a23 = asin(fabsf(areaParallelogram(p2 - c, p3 - c)) / rad2);

  float slope = (g2.z() - g1.z()) / a12;

  float rz3 = g2.z() + slope * a23;
  float delta_z = g3.z() - rz3;

  float cotTheta = slope * circle.curvature();
  float coshEta  = sqrt(1 + sqr(cotTheta));

  float pt = Bz / circle.curvature();
  float sigma_z = multipleScattering(pt, g3);

/*
  cerr << pt / coshEta
       << " dev = " << delta_z / (sigma_z * coshEta) 
       << " " << theBarrel << endl;
*/

  return (fabsf(delta_z / (sigma_z * coshEta)) < 5.);
}

/*****************************************************************************/
void ThirdHitRZPredictionWithHelix::initLayer(const DetLayer *layer)
{
  if ( layer->part() == barrel) {
    theBarrel = true;
    theForward = false;
    const BarrelDetLayer& bl = dynamic_cast<const BarrelDetLayer&>(*layer);
    float halfThickness  = bl.surface().bounds().thickness()/2;
    float radius = bl.specificSurface().radius();
    theDetRange = Range(radius-halfThickness, radius+halfThickness);
//    theTolerance = Margin(0.03,0.03);
  } else if ( layer->part() == forward) {
    theBarrel= false;
    theForward = true;
    const ForwardDetLayer& fl = dynamic_cast<const ForwardDetLayer&>(*layer);
    float halfThickness  = fl.surface().bounds().thickness()/2;
    float zLayer = fl.position().z() ;
    theDetRange = Range(zLayer-halfThickness, zLayer+halfThickness);
//    theTolerance = Margin(0.02,0.02);
  }
}
