#include "Utilities/Configuration/interface/Architecture.h"
#include "TrackerReco/LowPt/interface/ClusterShape.h"

#include "CommonDet/DetLayout/interface/DetLayer.h"

#include "CommonDet/BasicDet/interface/SimHit.h"
#include "TrackerReco/TkHitAssociation/interface/TkHitAssociator.h"

#include "TrackerReco/TkTrackingRegions/interface/GlobalTrackingRegion.h"
#include "TrackerReco/PixelTrackFit/interface/HitTripletPixelTrackBuilder.h"
#include "TrackerReco/PixelTrackFinder/interface/PixelRecTrack.h"

#include "TNtuple.h"
//extern TNtuple * clusShape;

float ClusterShape::limits[2][MaxSize + 1][2][MaxSize + 1][2][2][2];

/*****************************************************************************/
ClusterShape::ClusterShape()
{
  writeClusterShape = SimpleConfigurable<bool>(false,
                        "LowPt:writeClusterShape").value();
}

/*****************************************************************************/
ClusterShape::~ClusterShape()
{
}

/*****************************************************************************/
void ClusterShape::loadClusterShapeData()
{ 
 // Barrel
 {
  ifstream inFile;
  
  inFile.open("../data/barrel.par");
  cerr << " [ClusterShape] loading cluster shape data : barrel" << endl;
  
  while(inFile.eof() == false)
  { 
    int dx,dy;

    inFile >> dx;
    inFile >> dy;

    int sign = (dy < 0 ? 0 : 1);

    for(int i = 0; i<2 ; i++)
    for(int j = 0; j<2 ; j++)
    for(int k = 0; k<2 ; k++)
      inFile >> ClusterShape::limits[0][dx][sign][abs(dy)][i][j][k];
  }
  
  inFile.close();
 }
 
 // Forward
 {
  ifstream inFile;
  
  inFile.open("../data/forward.par");
  cerr << " [ClusterShape] loading cluster shape data : forward" << endl;
  
  while(inFile.eof() == false)
  { 
    int dx,dy;

    inFile >> dx;
    inFile >> dy;

    int sign = (dy < 0 ? 0 : 1);

    for(int i = 0; i<2 ; i++)
    for(int j = 0; j<2 ; j++)
    for(int k = 0; k<2 ; k++)
      inFile >> ClusterShape::limits[1][dx][sign][abs(dy)][i][j][k];
  }
  
  inFile.close();
 }
}

/*****************************************************************************/
int ClusterShape::getDirection(int low,int hig, int olow,int ohig)
{
  if(hig == ohig && low == olow) return  0;
  if(hig >= ohig && low >= olow) return  1; 
  if(hig <= ohig && low <= olow) return -1;
   
  return -2;
}
 
/*****************************************************************************/
bool ClusterShape::processColumn(pair<int,int> pos, bool inTheLoop)
{
  if(x[1] > -1)
  { // Process previous column
    if(low < y[0] || x[1] == x[0]) y[0] = low;
    if(hig > y[1] || x[1] == x[0]) y[1] = hig;

    if(x[1] > x[0])
    { // Determine direction
      int dir = getDirection(low,hig, olow,ohig);

      if(x[1] > x[0]+1)
      { // Check if direction changes
        if(odir*dir == -1)
        { odir = -2; return(false); }
      }

      if(x[1] <= x[0]+1 || odir == 0)
        odir = dir;

    }

    olow = low; ohig = hig;
  }
  else
  { // Very first column, initialize
   x[0] = pos.first;
  }

  // Open new column
  if(inTheLoop)
  {
    x[1] = pos.first;
    low  = pos.second;
    hig  = pos.second;
  }

  return(true);
}

/*****************************************************************************/
bool ClusterShape::determineShape(const RecHit *recHit)
{
 int nrows = pixelDet->specificTopology().nrows();
 int ncols = pixelDet->specificTopology().ncolumns();

 // Initialize
 bool isClean = true;
 bool atEdge  = false;

 x[1]=-1; olow=-2; ohig=-2; odir=0;
 pair<int,int> pos;

 // Process channels
 RecHit::ChannelContainer channels=recHit->channels();
 for(RecHit::ChannelIterator channel =channels.begin();
                             channel!=channels.end(); channel++)
 {
   // Position and charge
   pos = PixelDigi::channelToPixel(*channel);

   // Check if at the edge
   if(pos.first  == 0 || pos.first  == nrows-1 ||
      pos.second == 0 || pos.second == ncols-1)
   { atEdge = true; break; }

   if(pos.first > x[1])
   { // Process column
     if(processColumn(pos, true) == false)
     { isClean = false; break; }
   }
   else
   { // Increasing row
     if(pos.second > hig+1)
     { isClean = false; break; }

     hig = pos.second;
   }
 }

 // Process last column
 if(processColumn(pos, false) == false)
   isClean = false;

 return(isClean && !atEdge);
}

/*****************************************************************************/
bool ClusterShape::getFlipped(const PixelDet* pixelDet)
{
 if(pixelDet->type().part() == barrel)
 {
  float perp0 = pixelDet->toGlobal( Local3DPoint(0.,0.,0.) ).perp();
  float perp1 = pixelDet->toGlobal( Local3DPoint(0.,0.,1.) ).perp();
  if (perp1 < perp0) return 1;
                else return 0;
 }
 else
 {
  float rot = pixelDet->toGlobal( LocalVector (0.,0.,1.) ).z();
  float pos = pixelDet->toGlobal( Local3DPoint(0.,0.,0.) ).z();

  if(rot * pos < 0) return 1;
               else return 0;
 }
}

/*****************************************************************************/
bool ClusterShape::isInside(float a[2][2], float& wx, float& wy)
{
  if(wx > a[0][0]-0.2 && wx < a[0][1]+0.2 &&
     wy > a[1][0]-0.2 && wy < a[1][1]+0.2)
    return true;
  else 
    return false;
}

/*****************************************************************************/
void ClusterShape::compareWithSimHit
  (const RecHit *recHit, const PixelDet* pixelDet, int dx, int dy)
{
  TkHitAssociator theHitAssociator;
  vector<const SimHit*> simHits = theHitAssociator(*recHit);

  if(simHits.size() == 1)
  {
    const SimHit* simHit = *(simHits.begin());

    int part        = (pixelDet->type().part() == barrel ? 0 : 1);
    bool isFlipped  = getFlipped(pixelDet);
    float pitch_x   = pixelDet->specificTopology().pitch().first;
    float pitch_y   = pixelDet->specificTopology().pitch().second;
    float thickness = pixelDet->surface().bounds().thickness();

    float wx = simHit->localDirection().x() /
                 fabs(simHit->localDirection().z()) *
                   thickness / pitch_x;
    float wy = simHit->localDirection().y() /
                 fabs(simHit->localDirection().z()) *
                   thickness / pitch_y;

    vector<float> result;

    result.push_back(part); // part
    result.push_back(isFlipped == true ? 0 : 1); // type
    result.push_back(dx); // rx
    result.push_back(dy); // ry
    result.push_back(wx); // sx
    result.push_back(wy); // sy

//    clusShape->Fill(&result[0]);
  }
}

/*****************************************************************************/
bool ClusterShape::isCompatible
  (const RecHit *recHit, const LocalVector& dir)
{
 pixelDet = dynamic_cast<const PixelDet*>(&(recHit->det()));

 if(determineShape(recHit) == true) 
 { // Shape is definite, check compatibility with dir[]
   int dx = x[1] - x[0];
   int dy = y[1] - y[0];

   if(dx <= MaxSize && abs(dy) <= MaxSize)
   {
     if(odir != 0) dy *= odir;

     bool isFlipped = getFlipped(pixelDet);
     float pitch_x = pixelDet->specificTopology().pitch().first;
     float pitch_y = pixelDet->specificTopology().pitch().second;
  
     int sign = (dy < 0 ? 0 : 1);
     int part = (pixelDet->type().part() == barrel ? 0 : 1);
  
     // Compare rechit
     float c = pixelDet->surface().bounds().thickness()/fabs(dir.z()) *
               (isFlipped == false ? 1 : -1);

     // LocarVector
     float wx = dir.x()/pitch_x * c;
     float wy = dir.y()/pitch_y * c;

     bool ok =  isInside(limits[part][dx][sign][abs(dy)][0], wx,wy) ||
                isInside(limits[part][dx][sign][abs(dy)][1], wx,wy);

     // Compare simhit
     if(writeClusterShape)
       compareWithSimHit(recHit, pixelDet,dx,dy); 

     return ok;
   }
 }

 // Shape is indefinite or too wide
 return true;
}

/*****************************************************************************/
bool ClusterShape::checkPixelRecTrack
  (const vector<RecHit>& recHits, const vector<LocalVector>& localDirs)
{
 bool ok = true;

 vector<LocalVector>::const_iterator localDir = localDirs.begin();
 for(vector<RecHit>::const_iterator recHit  = recHits.begin();
                                    recHit != recHits.end(); recHit++) 
 {
   if(recHit->isValid())
   {
     if(isCompatible(&(*recHit), *localDir) == false)
     { ok = false; break; }
   }
   else
   { ok = false; break; }

   localDir++;
 }

 return(ok);
}
