#include "Utilities/Configuration/interface/Architecture.h"

#include "TrackerReco/LowPt/interface/LowPtTrackFinder.h"

#include "CommonReco/PatternTools/interface/TrajectorySeed.h"
#include "TrackerReco/TkSeedGenerator/interface/RegionalSeedGenerator.h"
#include "CommonReco/PatternTools/interface/BasicTrajectorySeed.h"
#include "CommonReco/PatternTools/interface/TrajectoryBuilder.h"
#include "CommonReco/TrackFitters/interface/KFTrajectorySmoother.h"
#include "CommonReco/PatternTools/interface/ConcreteRecTrack.h"
#include "CommonReco/Propagators/interface/GtfPropagator.h"
#include "CommonReco/KalmanUpdators/interface/KFUpdator.h"
#include "CommonReco/PatternTools/interface/TrajectoryCleaner.h"
#include "TrackerReco/GtfPattern/interface/RedundantSeedCleaner.h"
#include "Utilities/Notification/interface/TimingReport.h"

#include "CommonDet/PatternPrimitives/interface/MediumProperties.h"

#include "CARF/Reco/interface/ParameterSetBuilder.h"
#include "CARF/Reco/interface/ParameterSet.h"
#include "CARF/Reco/interface/ComponentSetBuilder.h"
#include "CARF/Reco/interface/ComponentSet.h"
#include "CARF/Reco/interface/ConfigAlgoFactory.h"
#include "CARF/Reco/interface/RecQuery.h"
#include "TrackerReco/TkPersistentTrack/interface/TkPRecTrack.h"

#include "TrackerReco/GtfPattern/interface/MaxHitsTrajectoryFilter.h"
 
/*****************************************************************************/
RecConfig LowPtTrackFinder::defaultConfig()
{
  // Parameters
  ParameterSetBuilder parameterBuilder;

  parameterBuilder.addParameter<bool>("seedCleaning",true); 
  parameterBuilder.addParameter<int>("minimumNumberOfHits",3);
  parameterBuilder.addParameter<int>("maxConsecLostHit",1);
  parameterBuilder.addParameter<int>("maxLostHit",3);

  parameterBuilder.addParameter("ptCut", 0.075, 0.0001);
  parameterBuilder.addParameter("mass",0.1057,0.0001);
  parameterBuilder.addParameter<bool>("createPersistableTracks",true);
  
  // Components
  ComponentSetBuilder componentBuilder;

  RecQuery seedQuery("RegionalPixelTripletSeedGenerator");
  componentBuilder.addComponent("SeedGenerator", seedQuery);

  RecQuery builderQuery("CombinatorialTrajectoryBuilder");
  componentBuilder.addComponent("TrajectoryBuilder", builderQuery);
  
  RecQuery smootherQuery("KFFittingSmoother");
  RecQuery splitFitQuery("KFSplittingFitter");
  smootherQuery.setComponent("Fitter",splitFitQuery);
  componentBuilder.addComponent("Smoother",smootherQuery);
  
  RecQuery cleanerQuery("TrajectoryCleanerBySharedHits");
  componentBuilder.addComponent("TrajectoryCleaner",cleanerQuery);

  return RecConfig("LowPtTrackFinder","0.1",
		   parameterBuilder.result(), componentBuilder.result());
}

/*****************************************************************************/
LowPtTrackFinder::LowPtTrackFinder(const RecConfig& config) : 
  ConfigAlgorithm(config)
{
  componentsInit();
  theSeedClean = parameter<bool>("seedCleaning");
}

/*****************************************************************************/
LowPtTrackFinder::~LowPtTrackFinder()
{
  delete theSeedGenerator;
  delete theTrajectoryBuilder;
  delete theTrajectorySmoother;
  delete theTrajectoryCleaner;
}


/*****************************************************************************/
vector<RecTrack *> LowPtTrackFinder::tracks(const TrackingRegion& region) const
{
  typedef SeedGenerator::SeedContainer::const_iterator        SeedIterator;

  bool persist = parameter<bool>("createPersistableTracks");
  vector<RecTrack *> result;
  try {
    
    SeedGenerator::SeedContainer sc = theSeedGenerator->seeds(region);
    
    vector<Trajectory> rawResult;
    int cleanSeed = 0;
    for ( SeedIterator is = sc.begin(); is != sc.end(); is++) {
      const TrajectorySeed& seed(*is);
      if( !theSeedClean || RedundantSeedCleaner::clean(seed, rawResult)){
	cleanSeed++;
	TrajectoryBuilder::TrajectoryContainer tc = 
	  theTrajectoryBuilder->trajectories(seed);
	theTrajectoryCleaner->clean(tc);
	for (TrajectoryBuilder::TrajectoryIterator it = tc.begin();
	it != tc.end(); it++) {
	  if((*it).isValid()) {  
	    rawResult.push_back( *it);
	  }
	}
      }
    }
    
    // clean the result
    vector<Trajectory> unsmoothedResult;
    theTrajectoryCleaner->clean(rawResult);

    for (TrajectoryBuilder::TrajectoryIterator itraw = rawResult.begin();
    itraw != rawResult.end(); itraw++) {
      if((*itraw).isValid()) unsmoothedResult.push_back( *itraw);
    }
    
    // smooth the surviving trajectories
    vector<Trajectory> smoothedResult;
    for (TrajectoryBuilder::TrajectoryIterator it = unsmoothedResult.begin();
    it != unsmoothedResult.end(); it++) {
      TrajectorySmoother::TrajectoryContainer smoothed =
	theTrajectorySmoother->trajectories(*it);
      smoothedResult.insert( smoothedResult.end(), 
			     smoothed.begin(), smoothed.end());
    }
    
   // convert to ConcreteRecTracks
    for (TrajectoryBuilder::TrajectoryIterator i = smoothedResult.begin();
    i != smoothedResult.end(); i++) {
      TTrack* track;
      if (persist) track = new TkPRecTrack(ConcreteRecTrack(*i));
      else         track = new ConcreteRecTrack(*i);
      if(track->impactPointState().isValid())
      {
        RecTrack * recTrack = new RecTrack(*track);
        result.push_back(recTrack);
//        delete track;
      }
    }
  } 
  catch( DetException& err) {
    cout << "[LowPtTrackFinder] got a DetException: " << err.what() << endl;
    cout << "[LowPtTrackFinder] will return an empty track container" << endl;
  }
  return result;
}

/*****************************************************************************/
void LowPtTrackFinder::componentsInit( )
{
  theSeedGenerator = createAlgo<RegionalSeedGenerator>(component("SeedGenerator"));

  RecConfig& builderConfig = const_cast<RecConfig&>(component("TrajectoryBuilder"));
  try {
    builderConfig.parameters().set("mass",parameter<double>("mass"));
    builderConfig.parameters().set("minimumNumberOfHits",parameter<int>("minimumNumberOfHits"));

    builderConfig.parameters().set("maxConsecLostHit",parameter<int>("maxConsecLostHit"));
    builderConfig.parameters().set("maxLostHit",parameter<int>("maxLostHit"));

    builderConfig.parameters().set("ptCut",parameter<double>("ptCut"));
  }
  catch ( Genexception& e ) {
    cout << "Exception when setting mass in LowPtTrackFinder: " << e.what() << endl;
  }
  theTrajectoryBuilder = createAlgo<TrajectoryBuilder>(builderConfig);

  // create ConfigAlgorithm

  RecConfig& smootherConfig = const_cast<RecConfig&>(component("Smoother"));
  try 
    {
      smootherConfig.parameters().set("mass",parameter<double>("mass"));
    }
  catch ( Genexception& e ) 
    {
      cout << "Exception when setting mass in LowPtTrackFinder: " << e.what() << endl;
    }
  
  // create ConfigAlgorithm
  theTrajectorySmoother = createAlgo<TrajectorySmoother>(smootherConfig);
  
  // create TrajectoryCleaner
  theTrajectoryCleaner = createAlgo<TrajectoryCleaner>(component("TrajectoryCleaner"));
}

/*****************************************************************************/
#include "Utilities/Notification/interface/PackageInitializer.h"
#include "CARF/Reco/interface/ConfigBuilder.h"

namespace {
  PKBuilder< ConfigBuilder<LowPtTrackFinder> > a("LowPtTrackFinderBuilder");
}
