#include "Utilities/Configuration/interface/Architecture.h"
#include "Utilities/Notification/interface/Observer.h"

// SimEvent
#include "TrackerReco/TkEvent/interface/TkRecEventFromCarf.h"
#include "TrackerReco/TkEvent/interface/TkSimEvent.h"

// RecQuery
#include "CARF/Reco/interface/RecQuery.h"
#include "CARF/Reco/interface/RecConfig.h"

// Algo
#include "CARF/Reco/interface/RecAlgorithm.h"
#include "CARF/Reco/interface/ConfigAlgorithm.h"
#include "CARF/Reco/interface/ConfigAlgoFactory.h"

#include <CARF/BaseSimEvent/interface/BaseSimEvent.h>
#include <CARF/SimCrossing/interface/SimSignal.h>

// CMS Tracker
#include "Tracker/TkLayout/interface/FullTracker.h"
#include "Tracker/TkLayout/interface/CmsTracker.h"

#include "CommonDet/BasicDet/interface/Det.h"
#include "CommonDet/BasicDet/interface/DetUnit.h"
#include "CommonDet/BasicDet/interface/DetType.h"
#include "CommonDet/BasicDet/interface/RecHit.h"
#include "CommonDet/DetLayout/interface/DetLayer.h"

// PixelTrackFinder
#include "CommonReco/PatternTools/interface/RecTrack.h"
#include "TrackerReco/PixelTrackFinder/interface/PixelRecTrack.h"
#include "TrackerReco/PixelTrackFinder/interface/PixelTrackFinderReco.h"
#include "TrackerReco/TkTrackingRegions/interface/GlobalTrackingRegion.h"

// Efficiency
#include <TrackerReco/TkEvent/interface/TkSimTrack.h>
#include <TrackerReco/TkEvent/interface/TkSimVertex.h>
#include <CommonDet/BasicDet/interface/SimHit.h>

// LowPtTracker
#include "TrackerReco/LowPt/interface/ClusterShape.h"
#include "TrackerReco/LowPt/interface/LowPtTrackFinder.h"

// Iostream
#include <iostream>
#include <fstream>

///////////////////////////////////////////////////////////////////////////////
class TestLowPt : public Observer<TkSimEvent*>
{
 public:
   TestLowPt();
   ~TestLowPt();

   virtual void upDate  (TkSimEvent* simEvent) 
                               { if (simEvent!=0) analysis(simEvent); }
   virtual void analysis(TkSimEvent* simEvent);

 private:
   float ptMin,originRadius,originHalfLength,originZPos;
   bool precise;

   PixelTrackFinderReco * thePTFReco; 

   int events;
};

///////////////////////////////////////////////////////////////////////////////
TestLowPt::TestLowPt() : thePTFReco(0)
{
  init();

  // Events
  events = 0;

  // Cylinder of origin
  ptMin            =
    SimpleConfigurable<float>(0.175, "LowPt:ptMin").value();
  originRadius     =
    SimpleConfigurable<float>(0.1, "LowPt:originRadius").value();
  originHalfLength =
    SimpleConfigurable<float>(0.1, "LowPt:originHalfLength").value();
  originZPos       = 0.;
  precise          = true;

  // Load cluster shape data
  ClusterShape::loadClusterShapeData();
}

///////////////////////////////////////////////////////////////////////////////
TestLowPt::~TestLowPt()
{
  cerr << "\r  events read: " << events << endl;

  delete thePTFReco;
}

///////////////////////////////////////////////////////////////////////////////
void TestLowPt::analysis(TkSimEvent* simEvent)
{
  // Get event
  const G3EventProxy* ev = simEvent->G3eventProxy();

  // First call
  if(!thePTFReco)
  {
    RecQuery PTFR("PixelTrackFinderReco");
    thePTFReco = createAlgo<PixelTrackFinderReco>(PTFR);
  }

  // Event info
  cerr << "\r run " << ev->simSignal()->id().runNumber()
       << " event " << ev->simSignal()->id().eventInRun()
       << " pileups " << ev->crossing()->pileups().size();

  events++;
  
  // In Pb+Pb, get primary vertex from simulation
  originZPos = ev->simSignal()->embdVertex(0).position().z();
  cerr << " originZPos = " << originZPos << endl;

  // Region
  GlobalTrackingRegion region
   (ptMin,originRadius,originHalfLength,originZPos,precise);

  // TrackFinder
  vector<PixelRecTrack *> recTracks = thePTFReco->tracks(region);
  cerr << " rectracks = " << recTracks.size() << endl;
}
 
///////////////////////////////////////////////////////////////////////////////
#include "Utilities/Notification/interface/PackageInitializer.h"
PKBuilder<TestLowPt> eventAnalyser("TestLowPt");
 
