#include "Utilities/Configuration/interface/Architecture.h"
#include "TrackerReco/PixelTrackFinder/src/PTFAmbiguityResolver.h"
#include <algorithm>
#include "CARF/Reco/interface/ParameterSetBuilder.h"

using namespace PixelTrackFinder;

RecConfig PTFAmbiguityResolver::defaultConfig()
{
  ParameterSetBuilder pb;
//  pb.addParameter("deltaEtaCut",0.01, 1.e-4);
  pb.addParameter("deltaEtaCut",0.1, 1.e-4);
  pb.addParameter("diffRelPtCut",0.2, 1.e-4);
  return RecConfig("PTFAmbiguityResolver","1",pb.result());
}


PTFAmbiguityResolver::PTFAmbiguityResolver(const RecConfig& cfg)
  : ConfigAlgorithm(cfg)
{
  theDeltaEtaCut = parameter<double>("deltaEtaCut");
  theDiffRelPtCut = parameter<double>("diffRelPtCut");
}

void PTFAmbiguityResolver::resolve(Tracks & tracks) const
{
  sort(tracks.begin(),tracks.end(), Less());
  Tracks_Itr it1 = tracks.begin(); 
  while (it1 != tracks.end() ) {
    for (Tracks_Itr it2 = it1+1; *it1 && it2<tracks.end(); it2++) {
      if (!*it2) continue;
      if (    asinh((**it2).cotTheta())-asinh((**it1).cotTheta())
            > theDeltaEtaCut) break;
      switch ( compare(*it1, *it2) ) {
         case killFirst :
           delete *it1;
           *it1 = 0;
           break;
         case killSecond :
           delete *it2;
           *it2 = 0;
           break;
         case mergeTwo :
           *it2 = merge(*it1,*it2);
           *it1 = 0;
           break;
         case goAhead : default: break;
      }
    }
    if (0 == *it1) tracks.erase(it1); else it1++;
  }
}


bool PTFAmbiguityResolver::
    Less::operator()(const PixelTrack* a, const PixelTrack* b) const
{ return a->cotTheta() < b->cotTheta(); }

const PixelTrack* PTFAmbiguityResolver::merge(
    const PixelTrack* a, const PixelTrack* b) const
{
// temporary algorith, takes track with bigger pt, to be reimplemented 
  if (a->pT() > b->pT()) {
    delete b;
    return a;
  } else {
    delete a;
    return b;
  }
} 


PTFAmbiguityResolver::Action PTFAmbiguityResolver::compare(
    const PixelTrack* a, const PixelTrack* b) const
{
  int nshared = 0;
  const vector<RecHit> & hitsA = a->hits();
  const vector<RecHit> & hitsB = b->hits();
  vector<RecHit>::const_iterator ihA, ihB;
  for (ihA = hitsA.begin(); ihA != hitsA.end(); ihA++)
    for (ihB = hitsB.begin(); ihB != hitsB.end(); ihB++) 
      if ( (ihA->globalPosition() - ihB->globalPosition()).mag() < 3.e-4) 
          nshared++; 

  if (nshared >= 2) {
    if (a->hits().size() >= 3 && b->hits().size() >= 3 ) 
      return (a->chi2() > b->chi2()) ? killFirst : killSecond;
    else if (b->hits().size() >= 3)
      return killFirst;
    else 
      return killSecond;
  }
  else if (nshared >= 1) {
    if (a->hits().size() != b->hits().size()) 
      return (a->hits().size() < b->hits().size()) ? killFirst : killSecond;
    else if (    a->hits().size() >= 3 
              && a->charge()==b->charge()
              && fabs(a->pT()-b->pT())/b->pT() < theDiffRelPtCut )
      return (a->chi2() > b->chi2()) ? killFirst : killSecond;
    else if (    a->hits().size() == 2 )
      return (a->pT() < b->pT()) ? killFirst : killSecond;
    else 
      return goAhead;
  }
  else return goAhead;
}

#include "Utilities/Notification/interface/PackageInitializer.h"
#include "CARF/Reco/interface/ConfigBuilder.h"
namespace {
  PKBuilder< ConfigBuilder<PTFAmbiguityResolver> > ar("PTFAmbiguityResolverBuilder");
}

