#include "RecoPixelVertexing/PixelTriplets/interface/ClusterInfo.h"
#include "RecoPixelVertexing/PixelTriplets/interface/ClusterShape.h"

#include "FWCore/Framework/interface/ESHandle.h"
#include "FWCore/Framework/interface/Handle.h"
#include "Geometry/Records/interface/TrackerDigiGeometryRecord.h"
#include "Geometry/TrackerGeometryBuilder/interface/TrackerLayerIdAccessor.h"
#include "DataFormats/TrackerRecHit2D/interface/SiPixelRecHitCollection.h"

#include <fstream>

bool  ClusterInfo::isFirst = true;
float ClusterInfo::limits[2][MaxSize + 1][2][MaxSize + 1][2][2][2];

map<DetId, map<const SiPixelRecHit*,ClusterData> >
  ClusterInfo::data_;

/*****************************************************************************/
ClusterInfo::ClusterInfo
  (const SiPixelRecHitCollection& theHits, const edm::EventSetup& es)
{
  // Load data
  if(isFirst == true)
  {
    loadClusterLimits();
    isFirst = false;
  }

  // Get tracker geometry
  edm::ESHandle<TrackerGeometry> tracker;
  es.get<TrackerDigiGeometryRecord>().get(tracker);
  theTracker = tracker.product();

  // ProcessHits
  processHits(theHits);
}

/*****************************************************************************/
ClusterInfo::ClusterInfo()
{
}

/*****************************************************************************/
ClusterInfo::~ClusterInfo()
{
}

/*****************************************************************************/
void ClusterInfo::processHits(const SiPixelRecHitCollection& theHits)
{
  ClusterData data;
  ClusterShape theClusterShape;

  cerr << " [ClusterInfo::processHits]"
       << " dets=" << theHits.id_size() 
       << " hits=" << theHits.size() << endl;

  data_.clear();

  for(SiPixelRecHitCollection::id_iterator id = theHits.id_begin();
                                           id!= theHits.id_end(); id++)
  {
    const PixelGeomDetUnit* pixelDet =
      dynamic_cast<const PixelGeomDetUnit*> (theTracker->idToDet(*id));

    SiPixelRecHitCollection::range range = theHits.get(*id);

    for(SiPixelRecHitCollection::const_iterator recHit = range.first;
                                                recHit!= range.second; recHit++)
    {
      theClusterShape.getExtra(*pixelDet, *recHit, data);
      data_[*id][&(*recHit)] = data;
    }
  }
}

/*****************************************************************************/
ClusterData ClusterInfo::getExtra(const SiPixelRecHit* recHit)
{
  return data_[recHit->geographicalId()][recHit];
}

/*****************************************************************************/
void ClusterInfo::loadClusterLimits()
{
 ifstream inFile;
 string fileNames[2] =
  {"/afs/cern.ch/user/s/sikler/public/data/barrel.par",
   "/afs/cern.ch/user/s/sikler/public/data/forward.par"};

 for(int part=0; part<2; part++)
 {
  inFile.open(fileNames[part].c_str());
  cerr << "  loading cluster shape data: "
       << fileNames[part].c_str() << endl;

  while(inFile.eof() == false)
  {
    int dx,dy;

    inFile >> dx;
    inFile >> dy;

    int sign = (dy < 0 ? 0 : 1);

    for(int i = 0; i<2 ; i++)
    for(int j = 0; j<2 ; j++)
    for(int k = 0; k<2 ; k++)
      inFile >> ClusterInfo::limits[part][dx][sign][abs(dy)][i][j][k];
  }

  inFile.close();
 }
}

/*****************************************************************************/
bool ClusterInfo::isInside(float a[2][2], pair<float,float> movement)
{
  if(movement.first  > a[0][0]-0.2 && movement.first  < a[0][1]+0.2 &&
     movement.second > a[1][0]-0.2 && movement.second < a[1][1]+0.2)
    return true;
  else
    return false;
}   

/*****************************************************************************/
bool ClusterInfo::isCompatible
  (const SiPixelRecHit *recHit, const LocalVector& dir)
{
  ClusterData data = getExtra(recHit);

  int dx = data.size.first;
  int dy = data.size.second;

  if(data.isStraight && data.isComplete && dx <= MaxSize && abs(dy) <= MaxSize)
  {
    int part = (data.isInBarrel ? 0 : 1);
    int sign = (dy < 0 ? 0 : 1);

    int orient = (data.isNormalOriented ? 1 : -1);

    pair<float,float> movement;
    movement.first  = dir.x() / (fabs(dir.z()) * data.tangent.first ) * orient;
    movement.second = dir.y() / (fabs(dir.z()) * data.tangent.second) * orient;

    return (isInside(limits[part][dx][sign][abs(dy)][0], movement) ||
            isInside(limits[part][dx][sign][abs(dy)][1], movement));
  }
  else
  {
    // Shape is not straight or not complete or too wide
    return true;
  }
}

/*****************************************************************************/
bool ClusterInfo::checkTrack
  (vector<const TrackingRecHit*> recHits, vector<LocalVector> localDirs)
{
  bool ok = true;

  vector<LocalVector>::const_iterator localDir = localDirs.begin();
  for(vector<const TrackingRecHit*>::const_iterator recHit = recHits.begin();
                                                    recHit!= recHits.end();
                                                    recHit++)
  {
    const SiPixelRecHit* pixelRecHit =
      dynamic_cast<const SiPixelRecHit *>(*recHit);

    if(pixelRecHit->isValid())
    {
      if(isCompatible(pixelRecHit, *localDir) == false)
      { ok = false; break; }
    }
    else
    { ok = false; break; }
 
    localDir++;
  }

  return ok;
}

