#include "FWCore/Framework/interface/EDAnalyzer.h"
#include "FWCore/Framework/interface/Event.h"
#include "FWCore/Framework/interface/EventSetup.h"
#include "FWCore/Framework/interface/Handle.h"
#include "FWCore/ParameterSet/interface/ParameterSet.h"

#include "SimTracker/TrackerHitAssociation/interface/TrackerHitAssociator.h"
#include "SimDataFormats/TrackingHit/interface/PSimHit.h"
#include "DataFormats/TrackerRecHit2D/interface/SiPixelRecHitCollection.h"
#include "RecoPixelVertexing/PixelTriplets/interface/ClusterData.h"
#include "RecoPixelVertexing/PixelTriplets/interface/ClusterInfo.h"

#include "TROOT.h"
#include "TFile.h"
#include "TNtuple.h"

#include <utility>
#include <vector>
#include <fstream>
using namespace std;

/*****************************************************************************/
class ClusterAnalyzer : public edm::EDAnalyzer
{
 public:
   explicit ClusterAnalyzer(const edm::ParameterSet& pset);
   ~ClusterAnalyzer();
   virtual void beginJob(const edm::EventSetup& es) { }
   virtual void analyze(const edm::Event& ev, const edm::EventSetup& es);
   virtual void endJob() { }

 private:
   map<int, map<int, map<int, map<int, map<int, vector<float> > > > > > scatter; 
   TFile* file;
   TNtuple* ntuple;
};

/*****************************************************************************/
ClusterAnalyzer::ClusterAnalyzer(const edm::ParameterSet& pset)
{
  file = new TFile("clusterShape.root","RECREATE");
  file->cd();
  ntuple = new TNtuple("clusterShape","clusterShape",
                       "subdet:branch:ex:ey:mx:my");

  scatter.clear();
}

/*****************************************************************************/
ClusterAnalyzer::~ClusterAnalyzer()
{
  file->cd();
  ntuple->Write();
  file->Close();

  for(int subdet=0; subdet<2 ; subdet++)
  {
    ofstream fileLimit;

    if(subdet == 0) fileLimit.open("barrel.par");
               else fileLimit.open("endcap.par");

    for(int ex= 0 ; ex<=20; ex++)
    for(int ey=-20; ey<=20; ey++)
    {
      fileLimit << " " << ex
                << " " << ey;

      for(int branch=-1; branch<=1; branch+=2)
      for(int dir=0; dir<2; dir++)
      {
        vector<float> pos = scatter[subdet][ex][ey][branch][dir];

        if(pos.size() >= 2)
        {
          // Cut off lower and upper 1%
          sort(pos.begin(),pos.end());
          int nElem = (int)(pos.size() * 0.05);
      
          float low  = *(pos.begin() + nElem);
          float high = *(pos.end()-1 - nElem);
          fileLimit << " " << low 
                    << " " << high;
        }
        else
          fileLimit << " 0. 0.";
      }

      fileLimit << endl;
    }

    fileLimit.close();
  }
}

/*****************************************************************************/
void ClusterAnalyzer::analyze(const edm::Event& ev, const edm::EventSetup& es)
{
  // Get pixel hits
  edm::Handle<SiPixelRecHitCollection> pixelHits;
  ev.getByType(pixelHits);
  const SiPixelRecHitCollection* theHits = pixelHits.product();

  // Get cluster info
  ClusterInfo theClusterInfo(*pixelHits,es);

  // Get associator
  TrackerHitAssociator theHitAssociator(ev);

  // Take all units
  for(SiPixelRecHitCollection::id_iterator id = theHits->id_begin();
                                           id!= theHits->id_end(); id++)
  {
    SiPixelRecHitCollection::range range = theHits->get(*id);

    // Take all hits
    for(SiPixelRecHitCollection::const_iterator recHit = range.first;
                                                recHit!= range.second; recHit++)
    {
      ClusterData data = theClusterInfo.getExtra(&(*recHit));
      vector<PSimHit> simHits = theHitAssociator.associateHit(*recHit);
      
      // Cluster should be straight, complete and only one association
      if(data.isStraight && data.isComplete && simHits.size() == 1)
      { 
        const PSimHit* simHit = &(simHits[0]);
 
        // Momentum above 50 MeV/c 
        if(simHit->momentumAtEntry().perp() > 0.050)
        {
          int sign = (data.isNormalOriented ? 1 : -1);

          pair<float,float> move;
          move.first  = simHit->localDirection().x() * sign /
                  (fabs(simHit->localDirection().z()) * data.tangent.first );
          move.second = simHit->localDirection().y() * sign /
                  (fabs(simHit->localDirection().z()) * data.tangent.second);
  
          int subdet = data.isInBarrel ? 0 : 1;
          int branch = sign * (simHit->localDirection().y() > 0 ? 1 : -1);
  
          vector<float> result;
          result.push_back(subdet);
          result.push_back(branch);
          result.push_back(data.size.first );
          result.push_back(data.size.second);
          result.push_back(move.first );
          result.push_back(move.second);
  
          ntuple->Fill(&result[0]);

          scatter[subdet][data.size.first]
                 [data.size.second][branch][0].push_back(move.first);
          scatter[subdet][data.size.first]
                 [data.size.second][branch][1].push_back(move.second);
        }
      }
    }
  }
}

DEFINE_FWK_MODULE(ClusterAnalyzer)
